 

/************************************************************************************************************
 **                                                                                                        **
 **             Examples1 :  Test ET-MINI MCP23017 Input(SW)/Output(LED) + INTB Of GPB                     **     
 **                          For "CP-JR ARM7 LPC2138" Board                                                **               
 **                                                                                                        ** 
 **                       :  Interface I2C (Use I/O Pin)                                                   ** 
 **                                                                                                        **
 ************************************************************************************************************
 **                                                                                                        **
 ** Target MCU  : LPC2138                                                                                  **
 **       	    : X-TAL : 19.6608 MHz                                                                      ** 
 **             : PLL Setup = M(3),P(2)	                               	                                   **
 **             : Run Speed 58.9824 MHz(With PLL)(19.6608MHz x 3 = 58.9824 MHz                             **         
 **             : VPB Clock(pclk) = CPU Clock/2 = 58.9824MHz/2 = 29.4912 MHz                               **
 **             : CPU Clock(cclk) = 58.9824 MHz                                                            **           
 **                                                                                                        **
 ** Editor           : uVision3 (V4.60.6.8)                                                                **
 ** Compiler         : ARMCC                                                                               **
 **																		                                                                     **
 ** Create By        : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                                                **
 ** Last Update      : 22/October/2013                                                                     **
 **                                                                                                        **
 ** ET-MINI MCP23017 : Set Jumper PULL-UP SDA,SCL,RES to ENA                                               **
 **                    Set Jumper ADDRESS A2,A1,A0 = 0                                                     **
 **                                                                                                        **      
 ** Control MCP23017 : Interface I2C(ID:0100 000X = 0x40(Wr),0x41(Rd) By Set A2,A1,A0=0 )                  **  
 **                    Set Register IOCON : Use Address BANK0(BANK=0),Byte Mode(SEQOP=1)                   **
 **                    and Enable Address PIN A0,A1,A2                                                     **
 **	                                                                                                       **
 ** I2C Interface    :   <MCU LPC2138>	  	  	           <MINI-MCP23017>                                 ** 
 **                                                            				                                     **
 **                     P0.2 : SCK(out)       Connect          SCL                                         **
 **                     P0.3 : SDA(In/Out)    Connect          SDA                                         **  
 **                                                                                                        **
 **                                                                                                        **
 **                                                                                                        **
 **                                        5V                                                              **
 **                                       --+--                                                            **
 **                                         |                                                              **
 **                                         |              ID = 0x40(WR),41(RD)                            **   
 **                                      +--+--+-----+                                         5V          **
 **                                      |     |     |        MINI-MCP23017                   --+--        **
 **                                      |     |     |   +------------------+                   |          **
 **       5V                             \     \     |   |                  |                   |          **
 **      --+--                    4k7x2  /     /     |   |                  |     LED0    10k   |          **
 **        |                             \     \     +---+VCC          GPA0 +-----|<|---/\/\/\--+          **
 **        |          MCU ARM7_LPC2138   /     /         |               .  |                   |          **
 **        |           +------------+    |     |         |               .  |                   |          **
 **        +-----------+            |    |     |         |               .  |     LED7    10k   |          **
 **                    |            |SCK |     |         |             GPA7 +-----|<|---/\/\/\--+          ** 
 **                    |        P0.2|----+-----|-------->|SCL               |                              **
 **                    |            |          |         |                  |                              **
 **                    |            |SDA       |         |                  |                              **
 **                    |        P0.3|<---------+-------->|SDA               |                              **
 **                    |            |                    |                  |                              **
 **                    |            |                    |                  |      SW0 (Set FullUp GPB)    **
 **              +-----+            |                   -|INTA         GPB0 +------/ ----------+           **
 **              |     |            |                    |               .  |                  |           **
 **              |     |            | Signal INT.GPB<----|INTB           .  |                  |           **                     
 **              |     |            |     active '0'     |               .  |      SW7         |           **
 **            --+--   |            |                    |             GPB7 +------/ ----------+           ** 
 **             ---    +------------+               +----+ GND              |                  |           **
 **              -                                  |    |                  |                  |           **
 **                                                 |    |    A2  A1  A0    |                  |           **
 **                                                 |    +----+---+---+-----+                --+--         **
 **                                               --+--       |   |   |                       ---          **
 **                                                ---        |   |   |                        -           **
 **                                                 -         +---+---+                                    **
 **                                                               |                                        **
 **                                                             --+--                                      **
 **                                                              ---                                       **
 **                                                               -                                        **
 **                                                                                                        **
 ************************************************************************************************************/

#include "LPC213x.h" 	                             
#include "stdio.h"

/*****************************************************************
 **     Define Register Address (IOCON.BANK=0 [Default])        **
 *****************************************************************/

 #define IODIRA     0x00     //Set Direction PA : 1= Input ,0=Output
 #define IODIRB     0x01     //Set Direction PB

 #define IOCON      0x0A     //Set Config Reg.

 #define GPPUA      0x0C     //Set Pullup Input PA  : 1=Enable PullUp ,0=Disable PullUp
 #define GPPUB      0x0D     //Set Pullup Input PB

 #define GPIOA      0x12     //IN/OUT DATA PORTA
 #define GPIOB      0x13     //IN/OUT DATA PORTB
 
 #define GPINTENA   0x04     //Reg. Select PIN Enable Interrupt-On-Chang For GPA
 #define GPINTENB   0x05     //Reg. Select PIN Enable Interrupt-On-Chang For GPB
  
 #define DEFVALA    0x06     //Reg. Select Value-bit Compare for Interrupt-On-Chang  In GPA  
 #define DEFVALB    0x07     //Reg. Select Value-bit Compare for Interrupt-On-Chang  In GPB  
 
 #define INTCONA    0x08     //Reg. Select Compare PIN GPA current with vale Reg.DEFVALA or Previous Value PIN GPA   
 #define INTCONB    0x09     //Reg. Select Compare PIN GPB current with vale Reg.DEFVALA or Previous Value PIN GPB



//************** Define I2C Pin **************************


#define     SCL_HI()    IOSET0 = 0x00000004     //P0.2=SCK:1
#define     SCL_LO()	  IOCLR0 = 0x00000004     //P0.2=SCK:0

  
#define     SDA_HI()    IOSET0 = 0x00000008     //P0.3=SDA:1
#define     SDA_LO()	  IOCLR0 = 0x00000008     //P0.3=SDA:0

#define     SDA_IN()    IODIR0 &= 0xFFFFFFF7    //P0.3=Input          	  
#define     SDA_OUT()   IODIR0 |= 0x00000008    //P0.3=Output           	  

#define     SDA         IOPIN0				          //Referent SDA = IOPIN0 for P0.3(input)



 /***************************************************************
  **       Define Control Byte Address for MCP23017            **
  **     When A0,A1,A2 = 0     (0100 A2A1A0 X)                 **
  ***************************************************************/
   
 #define  CTRL_Byte_WR   0x40  
 #define  CTRL_Byte_RD   0x41 

/**********************************************************
 **                    Function Delay                    **
 **********************************************************/

  void delay(int cnt)
   {
     int i,j ;
      for(i=0;i<cnt;i++)
        for(j=0;j<cnt;j++) ;
   }

/***********************************************************
 **                     Initial Port                      **
 ***********************************************************/
 
 void Init_port(void)
  {
  //------ Set GPIO  PIN -------

  PINSEL1 = 0x00000000               ;     //Set Port0-PIN[P0..P31]= GPIO for I2C

 
 //------ Set Direction of GPIO PIN -------

  IODIR0  = 0xFFFFFFFF               ;	   //Set Port0-[P0..P31]=Output	for I2C
 
 //--------- Default Value Port0 ---------

  IOPIN0  = 0x00000000               ;     //Set Data Port0 = 0 For I2C
  IOPIN1  = 0x00000000               ;     //Set Data Port1 = 0 For LCD
 
  } 


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +                                                                                      +
  +                              I2C Function                                            +
  +                                                                                      +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


 /**********************************************
  **                 I2C Start                **
  **********************************************/

void I2C_Start(void)
{
   
   SDA_OUT()              ;  //Set P0.3 = output  
 
   SDA_HI()               ;	
   SCL_HI()               ;
   SDA_LO()               ;	 
   SCL_LO()               ;
}


 /**********************************************
  **                 I2C Stop                **
  **********************************************/

void I2C_Stop(void)
{

   SDA_OUT()             ;  //Set P0.3=Output      
  
   SDA_LO()              ;  
   SCL_HI()              ;	  
   SDA_HI()              ;
   
}


/***********************************************************************************
 **                       Function I2C Write  Data 8 bit                          **
 **                                                                               **
 ***********************************************************************************
 **  Parameter :             												                              **
 **              dat = Data for write 1Byte                                       **       
 ***********************************************************************************/


void I2C_WrByte(unsigned char dat)			  	
{
   unsigned char loop ;		

   for(loop=0;loop<8;loop++)                //Loop Write data 8 Bit 
    {			    		
		//--------- write Data(SDA:P0.3) ---------

	    SDA_OUT()                            ;  //Set P0.3=SDA= Output  

	    if((dat & 0x80)== 0x80)	                //Check data bit8 is 0 or 1
		    SDA_HI()                           ;  //Sent '1' Out Pin SDA(P0.3)
	    else
		    SDA_LO()                           ;	//Sent '0' Out Pin SDA(P0.3)

      dat <<= 1                            ;	//Shift data Next Bit 

	    SCL_HI()                             ;  //Clock HI     
      delay(10);			
	    SCL_LO()                             ;  //Clock LO   
      delay(10);
			
    } 	  	  

   SDA_IN()                                ;  //Set P0.3=Input 

   SCL_HI()                                ;  //Start ACK Clock
   while(SDA & 0x00000008){;}                 //Check ACK:P0.3 = 0 Out Loop
   SCL_LO()                                ;  //End ACK Clock
   	 
}



/*******************************************************************************
 **                 Function I2C Read  Data 8 bit                             **
 **                                                                           **           
 *******************************************************************************/

 unsigned char I2C_RdByte(void)			  	
 { 
   unsigned char loop,result=0 ;		

   for(loop = 0; loop < 8; loop++)	            //Loop Read data 8-Bit
    {
      result <<= 1                          ;   //Shift Result Save (MSB <- LSB)
    
      SDA_IN()                              ;      
      SCL_HI()                              ;   //Strobe Read SDA
   	  	  
	    if((SDA & 0x00000008) == 0x00000008)      //Check Data Read Pin SDA(PA0) is '0' or '1'
	       result |= 0x01                     ;   //If Read Pin SDA is '1' Give Save Bit Data = 1

      SCL_LO()                              ;   //Next Bit Read
    }
	 
  return result                             ;
}



/******************************************************************************* 
 **               Function I2C Write MCP23017 (Byte-Mode)  1 Sequence         **
 ** Parameter :                                                               **
 **            Control Byte Write MCP23017 = 0x40 (A2,A1,A0=0)                **   
 **            addr  = Register Address for write(1Byte)                      **
 **            dat   = Data to MCP23017                                       **    
 *******************************************************************************/
 
void I2C_WrMCP23017 (unsigned char addr,unsigned char dat)
{ 
  
   I2C_Start()                   ;

//---------- Sent Slave address for Write ---------------

   I2C_WrByte(CTRL_Byte_WR)      ;  //Send ID Code MCP32017+Write (0100 000+0) = 0x40
 
//------------ Sent address Register --------------
 
   I2C_WrByte(addr)              ;	//Send Address Reg(0x00-0x15). to MCP23017

//------------- Sent data to MCP23017 ---------------

   I2C_WrByte(dat)               ;	//Send Data Control to MCP23017
  
   I2C_Stop()                    ;
 
 
}



/******************************************************************************
 **              Function I2C Read MCP32017 (Byte-Mode) 1 Sequence           **
 **                                                                          **
 ** Parameter :                                                              **
 **             Control Byte Read MCP23017 = 0x41 (A2,A1,A0=0)               **
 **             addr = Register Address for Read (1Byte)                     **
 ******************************************************************************/


 unsigned char I2C_RdMCP23017(unsigned char addr)
  {
    unsigned char Rdx  ;
   
 //-------------- Sent Status Start for write -----------

   I2C_Start();

//------------- Sent Slave Address for Write------------	
  
   I2C_WrByte(CTRL_Byte_WR)    ;    	//Send ID Code MCP23017,Write (0100 000+0) = 0x40

//------------ Sent address Register --------------

   I2C_WrByte(addr)            ;     	//Send Address Reg. for Read MCP32017
 
//---------- New Sent Status Start For Read ----------

   I2C_Start()                 ;

//-------------Sent Slave address for Read ------------

   I2C_WrByte(CTRL_Byte_RD)    ;	    //Send ID Code MCP23017 ,Write (0100 000+1) = 0x41

//------------- Read data 1 Byte ------------
   
   Rdx = I2C_RdByte()          ;					
  
//---------- Sent Status Stop Red ---------------
  
   I2C_Stop()                  ; 
	 
 
   return Rdx                  ;        //Return Data

  }



/*****************************************
 **     Function Initial MCP23017       **
 *****************************************/

 void Init_MCP23017(void)
  {
   
    //---------------- Setup I/O Port --------------

    I2C_WrMCP23017(IOCON,0x28)      ;  //Config BANK0,Byte Mode,Enable MCP23017 AddressPin,INTB associated PGB

    I2C_WrMCP23017(IODIRA,0x00)     ;  //Set GPA = Output For LED
    I2C_WrMCP23017(IODIRB,0xFF)     ;  //Set GPB = Input For SW
   
    I2C_WrMCP23017(GPPUB,0xFF)      ;  //PullUp GPB Enable 


    //---------- Set Up Output Interrupt INTB ------------
 
    I2C_WrMCP23017(GPINTENB,0xFF)   ; // Enable GPB[0..7] pin For Interrupt-On-Chang
    
    I2C_WrMCP23017(DEFVALB,0x0F)    ; // Set Vale Compare to '1',Use 4 bit Low Compare with GPB[0..3] 
  
    I2C_WrMCP23017(INTCONB,0x0F)    ; // Set Bit GPB[0..3] Compare with Value Reg.DEFVALB ,And GPB[4..7] Compare with previous GPB[4..7] PIN

  }


/******************************************************************
 **		  												                                 **
 **                          Main Program                        **
 **  														 	                               **
 ******************************************************************/
 
int main(void)
 {  
		 unsigned char sw = 0      ;
		 
     Init_port()			         ;
     Init_MCP23017()           ;
 
     while(1)
	    {   
        sw = I2C_RdMCP23017(GPIOB)          ;   //Read Data from SW. GPB
         
		    if(sw != 0xFF)                          //Check Press SW Bit=0
		    {
		       I2C_WrMCP23017(GPIOA,sw)         ;   //On LED(Logic 0) bit at press sw.
		   
           do                                   //Protect Press SW.Rebeat or Double
			      {
             sw = I2C_RdMCP23017(GPIOB)     ;   //Read Data from SW. GPB
			      }while(sw != 0xFF)              ;   //Check Release SW.
		    
			  } 
		   else  //Not Press SW. (0xFF)
		    {
		      I2C_WrMCP23017(GPIOA,0xFF)        ;   //Clear LED OFF  (LED Active Low)
		    }	  
    
        delay(100)                          ;   //delay for Bout SW. 
     }
 }

